#include <stdio.h>
#include <string.h>
#include <stdlib.h>

/* change this value if you have more than 50 dimensions in your dataset */
#define MAX_DIMENSIONS  50

/* To keep the minimum and maximum values for the dimensions */
float minimumValue[MAX_DIMENSIONS];
float maximumValue[MAX_DIMENSIONS];

/* To keep the column names */
char *dimensionNamesArray[MAX_DIMENSIONS];
char *infileName;           // Input file
char *outfileName;          // Output file

int totalDimensions = 0;    // number of dimensions


/*
 * Initialize minimum and maximum values for dimensions.
 * If your application has the range of values above or below
 * the specified values, please change the initial values.
 */
void initialize(void)
{
    int index;

    for (index=0 ; index<MAX_DIMENSIONS ; index++)
    {
        minimumValue[index] = 9999;
        maximumValue[index] = -9999;
    }
}


/*
 * Prepares the header for OKC file format
 * This function will go through the file once to determine
 * number of dimensions, total records, minimum and maximum
 * values for the dimensions and print them in the format
 * required by OKC file.
 */
void prepareHeader(void)
{
    char dimensionNames[1000],   // read the names in a temporary storage
         dimensionValues[1000];  // read the value in a temporary storage
    char *separator = ",";  // separators required by strtok() function
    char *token;            // keep the tokens from the string read
    float value = 0;        // read the values of dimensions
    int i,
        count = 0;          // count of the number of records

    FILE *infile;           // input CSV format file
    FILE *outfile;          // output OKC format file

    initialize();           // intialize minimum and maximum values of dimensions
    infile = fopen(infileName, "r");
    outfile = fopen(outfileName, "w");

    if (infile == NULL)
    {
        printf("File does not exist\n");
        exit(0);
    }

    fscanf(infile, "%s\n", &dimensionNames); // read all the dimension names

    /* Establish string and get the first token: */
    token = strtok(dimensionNames, separator);  // get the first token name

    while(token != NULL)
    {
        dimensionNamesArray[totalDimensions] = (char *)malloc(sizeof(token));

        // write the dimension names in the OKC file
        strcpy(dimensionNamesArray[totalDimensions], token);

        // count the #dimensions
        totalDimensions++;

        // Get next token
        token = strtok( NULL, separator);
    }

    // find out the # data, minimum and maximum values for all the dimensions
    while (!feof(infile))
    {
        fscanf(infile, "%s\n", &dimensionValues);
        count++;
        token = strtok(dimensionValues, separator);

        for (i=0 ; i<totalDimensions && token!=NULL ; i++)
        {
            value = (float) atof(token);

            if (value < minimumValue[i])        // check for minimum value
                minimumValue[i] = value;
            if (value > maximumValue[i])        // check for maximum value
                maximumValue[i] = value;

            token = strtok( NULL, separator);
        }
    }

    /* Write the measured values to the output file */
    fprintf(outfile, "%d %d\n", totalDimensions, count);

    for (i=0 ; i<totalDimensions ; i++)
        fprintf(outfile, "%s\n", dimensionNamesArray[i]);

    for (i=0 ; i<totalDimensions ; i++)
        fprintf(outfile, "%7.6f %7.6f 3\n", minimumValue[i], maximumValue[i]);

    fclose(infile);
    fclose(outfile);
}


/*
 * Print data values from input file
 */
void printDataValues(void)
{
    char dimensionNames[1000],
         dimensionValues[1000];
    char *separator = ",";
    char *token;
    float value = 0;
    int i;

    FILE *infile;
    FILE *outfile;

    infile = fopen(infileName, "r");
    outfile = fopen(outfileName, "a");

    if (infile == NULL)
    {
        printf("File does not exist\n");
        exit(0);
    }

    // skip dimension names
    fscanf(infile, "%s\n", &dimensionNames); // read all the dimension names

    // Read the data and write it in space separated format required
    // by OKC file
    while (!feof(infile))
    {
        fscanf(infile, "%s\n", &dimensionValues);
        token = strtok(dimensionValues, separator);

        for (i=0 ; i<totalDimensions && token!=NULL ; i++)
        {
            value = (float) atof(token);
            fprintf(outfile, "%4.6f\t", value);
            token = strtok( NULL, separator);
        }
        fprintf(outfile, "\n");
    }

    fclose(infile);
    fclose(outfile);
}


void usage(void)
{
    printf("Usage : excel2xmdv inputFileName\n");
    exit(0);
}


void main(int argc, char *argv[])
{
    if (argc == 0)
        usage();
    infileName = argv[1];
    outfileName = (char *)malloc(sizeof(infileName));
    strcpy(outfileName, infileName);
	*strstr(outfileName, ".csv") = 0;
    outfileName = strcat(outfileName, ".okc");
    prepareHeader();
    printDataValues();
    printf("Your output .okc file is : %s !!!\n", outfileName);
}
